
'use client';

import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { useLanguage } from '@/contexts/language-context';
import { KeyRound, AreaChart, Combine, Signal, Users, Layers, Sparkles } from 'lucide-react';

const iconMap: { [key: string]: React.ElementType } = {
  'EPC & Turnkey Solutions': KeyRound,
  'Project Financing & Investment Structuring': AreaChart,
  'Energy Transition Projects': Combine,
  'Smart Field Digitalization': Signal,
  'Industrial R&D and Local Partner Programs': Users,
};

const cardConfig = [
  { 
    icon: KeyRound, 
    gradient: "from-blue-600 via-cyan-500 to-teal-400",
    bgGradient: "from-blue-50/80 via-cyan-50/60 to-teal-50/40",
  },
  { 
    icon: AreaChart, 
    gradient: "from-purple-600 via-pink-500 to-rose-400",
    bgGradient: "from-purple-50/80 via-pink-50/60 to-rose-50/40",
  },
  { 
    icon: Combine, 
    gradient: "from-emerald-600 via-green-500 to-lime-400",
    bgGradient: "from-emerald-50/80 via-green-50/60 to-lime-50/40",
  },
  { 
    icon: Signal, 
    gradient: "from-orange-600 via-amber-500 to-yellow-400",
    bgGradient: "from-orange-50/80 via-amber-50/60 to-yellow-50/40",
  },
  { 
    icon: Users, 
    gradient: "from-indigo-600 via-violet-500 to-fuchsia-400",
    bgGradient: "from-indigo-50/80 via-violet-50/60 to-fuchsia-50/40",
  }
];

export function Capabilities() {
  const { content, language } = useLanguage();
  const { capabilities } = content;
  const [hoveredValue, setHoveredValue] = useState<number | null>(null);

  return (
    <section id="capabilities" className="scroll-mt-20 relative overflow-hidden bg-gradient-to-br from-slate-50 via-white to-purple-50/30 py-20 px-4">
      {/* Animated Background Elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <motion.div
          className="absolute -top-20 -left-40 w-96 h-96 bg-purple-400/10 rounded-full blur-3xl"
          animate={{
            x: [0, 100, 0],
            y: [0, -50, 0],
            scale: [1, 1.2, 1],
          }}
          transition={{ duration: 20, repeat: Infinity, ease: "easeInOut" }}
        />
        <motion.div
          className="absolute -bottom-20 -right-40 w-96 h-96 bg-blue-400/10 rounded-full blur-3xl"
          animate={{
            x: [0, -100, 0],
            y: [0, 50, 0],
            scale: [1, 1.3, 1],
          }}
          transition={{ duration: 25, repeat: Infinity, ease: "easeInOut" }}
        />
        <motion.div
          className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[600px] h-[600px] bg-gradient-to-r from-orange-400/5 to-cyan-400/5 rounded-full blur-3xl"
          animate={{
            rotate: [0, 360],
            scale: [1, 1.1, 1],
          }}
          transition={{ duration: 30, repeat: Infinity, ease: "linear" }}
        />
      </div>

      <div className="relative z-10 max-w-7xl mx-auto">
        {/* Header Section */}
        <motion.div 
          className="mx-auto max-w-3xl text-center mb-20"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
        >
          <motion.div
            className="inline-flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-purple-600/10 to-blue-600/10 rounded-full mb-6 backdrop-blur-sm border border-purple-200/50"
            initial={{ scale: 0 }}
            whileInView={{ scale: 1 }}
            viewport={{ once: true }}
            transition={{ type: "spring", duration: 0.8 }}
          >
            <Layers className="w-4 h-4 text-purple-600" />
            <span className="text-sm font-semibold bg-gradient-to-r from-purple-600 to-blue-600 bg-clip-text text-transparent">
              {language === 'ar' ? 'قوتنا المشتركة' : 'Our Joint Power'}
            </span>
          </motion.div>

          <h2 className="font-bold tracking-tight text-slate-900 text-4xl sm:text-5xl mb-6 relative">
            <motion.span
              className="inline-block"
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.6, delay: 0.2 }}
            >
              {capabilities.title}
            </motion.span>
            <motion.div
              className="absolute -bottom-2 left-1/2 -translate-x-1/2 h-1 bg-gradient-to-r from-purple-600 via-blue-600 to-cyan-600 rounded-full"
              initial={{ width: 0 }}
              whileInView={{ width: "120px" }}
              viewport={{ once: true }}
              transition={{ duration: 0.8, delay: 0.4 }}
            />
          </h2>
          
           <p className="mt-8 text-lg leading-8 text-slate-600">
            {capabilities.subtitle}
          </p>

        </motion.div>
        
        {/* Capabilities Grid */}
        <div className="grid grid-cols-1 gap-8 md:grid-cols-2 lg:grid-cols-3">
          {capabilities.items.map((item, index) => {
            const config = cardConfig[index % cardConfig.length];
            const Icon = config.icon;
            const isHovered = hoveredValue === index;

            return (
              <motion.div
                key={item.title}
                initial={{ opacity: 0, scale: 0.8, y: 50 }}
                whileInView={{ opacity: 1, scale: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ 
                  duration: 0.6, 
                  delay: index * 0.1,
                  type: "spring",
                  stiffness: 100
                }}
                onMouseEnter={() => setHoveredValue(index)}
                onMouseLeave={() => setHoveredValue(null)}
                className="relative group perspective-1000"
              >
                {/* Glow Effect */}
                <motion.div
                  className={`absolute -inset-1.5 bg-gradient-to-r ${config.gradient} rounded-3xl blur-xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
                  animate={isHovered ? { scale: [1, 1.05, 1] } : {}}
                  transition={{ duration: 2, repeat: Infinity }}
                />

                {/* Card */}
                <motion.div
                  className={`relative h-full rounded-3xl overflow-hidden shadow-2xl transition-all duration-500 border-2 border-transparent group-hover:border-white/50`}
                  whileHover={{ 
                    y: -8,
                    rotateX: 5,
                    rotateY: 5,
                  }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  {/* Background with Gradient */}
                  <div className={`absolute inset-0 bg-gradient-to-br ${config.bgGradient} transition-all duration-500`} />
                  
                  {/* Animated Pattern Overlay */}
                  <motion.div
                    className="absolute inset-0 opacity-10"
                    style={{
                      backgroundImage: `radial-gradient(circle at 50% 50%, currentColor 1px, transparent 1px)`,
                      backgroundSize: '20px 20px',
                    }}
                    animate={{
                      backgroundPosition: ['0px 0px', '20px 20px'],
                    }}
                    transition={{ duration: 20, repeat: Infinity, ease: "linear" }}
                  />

                  {/* Gradient Overlay */}
                  <motion.div
                    className={`absolute inset-0 bg-gradient-to-br ${config.gradient} opacity-0 group-hover:opacity-20 transition-opacity duration-500`}
                  />

                  {/* Content */}
                  <div className="relative h-full flex flex-col items-center text-center p-8 z-10">
                    {/* Icon */}
                    <motion.div
                      className={`w-20 h-20 rounded-2xl bg-gradient-to-br ${config.gradient} flex items-center justify-center shadow-xl mb-6`}
                      whileHover={{ 
                        scale: 1.1,
                        rotate: [0, -5, 5, -5, 0]
                      }}
                      transition={{ duration: 0.5 }}
                    >
                      <Icon className="w-10 h-10 text-white" />
                    </motion.div>

                    {/* Text Content */}
                    <div>
                      <h3 className="text-xl font-bold text-slate-900 mb-3 drop-shadow-sm">
                        {item.title}
                      </h3>
                      <motion.p
                        className="text-base text-slate-700 leading-relaxed"
                        initial={{ opacity: 0.7 }}
                        animate={{ opacity: isHovered ? 1 : 0.7 }}
                        transition={{ duration: 0.3 }}
                      >
                        {item.description}
                      </motion.p>
                    </div>

                    {/* Floating particles */}
                    {isHovered && (
                      <>
                        {[...Array(5)].map((_, i) => (
                          <motion.div
                            key={i}
                            className={`absolute w-2 h-2 rounded-full bg-gradient-to-r ${config.gradient}`}
                            initial={{
                              x: Math.random() * 200 - 100,
                              y: Math.random() * 200 - 100,
                              opacity: 0,
                              scale: 0,
                            }}
                            animate={{
                              y: [0, -100],
                              opacity: [0, 1, 0],
                              scale: [0, 1, 0],
                            }}
                            transition={{
                              duration: 2,
                              delay: i * 0.2,
                              repeat: Infinity,
                            }}
                            style={{
                              left: '50%',
                              top: '50%',
                            }}
                          />
                        ))}
                      </>
                    )}
                  </div>
                </motion.div>
              </motion.div>
            );
          })}
        </div>
      </div>
    </section>
  );
}
